/*
 * $Id: ReflectionUtils.java 5263 2009-03-09 13:36:43Z uckelman $
 *
 * Copyright (c) 2008 by Joel Uckelman
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License (LGPL) as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, copies are available
 * at http://www.opensource.org.
 */
package VASSAL.tools;

import java.lang.reflect.InvocationTargetException;

import VASSAL.tools.logging.Logger;

/**
 * @author Joel Uckelman
 * @since 3.1.0
 */
public class ReflectionUtils {
  private ReflectionUtils() {}

  public static void handleImportClassFailure(Throwable t, String className) {
    handle(
      t,
      className,
      // thrown by Class.forName()
      ClassNotFoundException.class,
      ExceptionInInitializerError.class,
      LinkageError.class,
      // thrown by Class.getConstructor()
      NoSuchMethodException.class,
      SecurityException.class,
      // thrown by Constructor.newInstance()
      IllegalAccessException.class,
      IllegalArgumentException.class,
      InstantiationException.class,
      InvocationTargetException.class
    );
  }

  /**
   * Handle a {@link Throwable} generated by
   * <code>Class.getConstructor().newInstance()</code>.
   *
   * @param t the <code>Throwable</code>
   */ 
  public static void handleNewInstanceFailure(Throwable t, Class<?> c) {
    handle(
      t,
      c.getName(),
      IllegalAccessException.class,
      IllegalArgumentException.class,
      InstantiationException.class,
      InvocationTargetException.class,
      NoSuchMethodException.class,
      SecurityException.class,
      ExceptionInInitializerError.class
    );
  }

  private static void handle(Throwable t, String className, Class<?>... args) {
    // find and rethrow causes which are not bugs
    ThrowableUtils.throwRecent(OutOfMemoryError.class, t);

    // always a bug if the class was from the VASSAL.* hiearchy
    if (className.startsWith("VASSAL.")) {
      ErrorDialog.bug(t);
      return;
    }
    
    // these errors are expected from custom classes
    for (Class<?> cl : args) {
      if (cl.isInstance(t)) {
        ErrorDialog.showDetailsDisableable(
          t,
          ThrowableUtils.getStackTrace(t),
          className,
          "Error.custom_class_error",
          className
        );
        return;
      }
    }
  
    // otherwise something weird happened
    if (t instanceof Error) {
      // some unusual problem occurred    
      throw (Error) t;
    }
    else if (t instanceof RuntimeException) {
      // some unusual problem occurred    
      throw (RuntimeException) t;
    }
    else {
      // this should never happen
      throw new IllegalStateException(t);
    }
  }
}
